import React, { useCallback, useEffect, useMemo, useRef, useState } from "react";
import StoryboardControls from "./components/StoryboardControls";
import StoryboardGrid from "./components/StoryboardGrid";
import VideoUploader from "./components/VideoUploader";
import type { FrameSnapshot, VideoMeta } from "./types";

const SUPPORTED_TYPES = ["video/mp4", "video/webm"] as const;
const MAX_FILE_SIZE_MB = 200;
const DEFAULT_STEP_SECONDS = 1.5;

const App: React.FC = () => {
  const videoRef = useRef<HTMLVideoElement | null>(null);
  const canvasRef = useRef<HTMLCanvasElement | null>(null);

  const [file, setFile] = useState<File | null>(null);
  const [videoUrl, setVideoUrl] = useState<string | null>(null);
  const [frames, setFrames] = useState<FrameSnapshot[]>([]);
  const [stepSeconds, setStepSeconds] = useState<number>(DEFAULT_STEP_SECONDS);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [statusMessage, setStatusMessage] = useState<string | null>(null);
  const [videoMeta, setVideoMeta] = useState<VideoMeta | null>(null);
  const [downloadFormat, setDownloadFormat] = useState<
    "image/jpeg" | "image/png"
  >("image/jpeg");

  useEffect(() => {
    if (!videoUrl) return;
    return () => URL.revokeObjectURL(videoUrl);
  }, [videoUrl]);

  const helperText = useMemo(() => {
    return `Максимальный размер файла: ${MAX_FILE_SIZE_MB} МБ.`;
  }, []);

  const handleFileSelected = (selectedFile: File) => {
    setError(null);
    setFrames([]);
    setVideoMeta(null);

    if (!SUPPORTED_TYPES.includes(selectedFile.type as (typeof SUPPORTED_TYPES)[number])) {
      setError("Неподдерживаемый формат. Используйте MP4 или WebM.");
      return;
    }

    const sizeMb = selectedFile.size / (1024 * 1024);
    if (sizeMb > MAX_FILE_SIZE_MB) {
      setError(`Файл слишком большой. Лимит ${MAX_FILE_SIZE_MB} МБ.`);
      return;
    }

    setFile(selectedFile);
    setVideoUrl(URL.createObjectURL(selectedFile));
  };

  const handleLoadedMetadata = () => {
    if (!videoRef.current) return;
    setVideoMeta({
      duration: videoRef.current.duration,
      width: videoRef.current.videoWidth,
      height: videoRef.current.videoHeight,
    });
  };

  const formatTimestamp = useCallback((seconds: number) => {
    const totalSeconds = Math.floor(seconds);
    const minutes = Math.floor(totalSeconds / 60);
    const remainder = totalSeconds % 60;
    return `${minutes}:${remainder.toString().padStart(2, "0")}`;
  }, []);

  const captureFrameAt = useCallback(
    (time: number) => {
      return new Promise<FrameSnapshot>((resolve, reject) => {
        const video = videoRef.current;
        const canvas = canvasRef.current;
        if (!video || !canvas) {
          reject(new Error("Видео или canvas недоступны."));
          return;
        }

        const handleSeeked = () => {
          const context = canvas.getContext("2d");
          if (!context) {
            reject(new Error("Не удалось получить контекст canvas."));
            return;
          }

          // Важно: размер canvas должен совпадать с размером видео,
          // иначе получим искажения при отрисовке.
          canvas.width = video.videoWidth;
          canvas.height = video.videoHeight;
          context.drawImage(video, 0, 0, canvas.width, canvas.height);

          // Используем JPEG для превью: меньше вес, быстрее обработка.
          const dataUrl = canvas.toDataURL("image/jpeg", 0.92);
          const id = `${time}-${Math.random().toString(36).slice(2)}`;
          resolve({ id, time, dataUrl, format: "image/jpeg" });
        };

        const handleError = () => {
          reject(new Error("Не удалось перемотать видео."));
        };

        video.removeEventListener("seeked", handleSeeked);
        video.removeEventListener("error", handleError);
        video.addEventListener("seeked", handleSeeked, { once: true });
        video.addEventListener("error", handleError, { once: true });
        video.currentTime = time;
      });
    },
    []
  );

  const generateStoryboard = useCallback(async () => {
    if (!videoRef.current || !file) {
      setError("Сначала загрузите видеофайл.");
      return;
    }

    const safeStep = Math.max(0.2, stepSeconds || DEFAULT_STEP_SECONDS);
    setStepSeconds(safeStep);

    setIsLoading(true);
    setError(null);
    setStatusMessage("Подготавливаем видео…");
    setFrames([]);

    const video = videoRef.current;
    video.pause();

    if (!video.duration || Number.isNaN(video.duration)) {
      setError("Не удалось получить длительность видео.");
      setIsLoading(false);
      return;
    }

    const totalFrames = Math.ceil(video.duration / safeStep);
    const capturedFrames: FrameSnapshot[] = [];

    for (let index = 0; index < totalFrames; index += 1) {
      const time = Math.min(index * safeStep, video.duration);
      setStatusMessage(
        `Извлекаем кадр ${index + 1} из ${totalFrames} (${formatTimestamp(
          time
        )})`
      );

      try {
        const frame = await captureFrameAt(time);
        capturedFrames.push(frame);
      } catch (err) {
        setError(
          err instanceof Error
            ? err.message
            : "Ошибка при извлечении кадров."
        );
        break;
      }
    }

    setFrames(capturedFrames);
    setStatusMessage(null);
    setIsLoading(false);
  }, [captureFrameAt, file, formatTimestamp, stepSeconds]);

  const handleDownload = useCallback(
    (frame: FrameSnapshot) => {
      const link = document.createElement("a");
      const extension = downloadFormat === "image/png" ? "png" : "jpg";
      link.download = `frame-${Math.round(frame.time)}s.${extension}`;

      if (downloadFormat === frame.format) {
        link.href = frame.dataUrl;
      } else {
        const canvas = canvasRef.current;
        const context = canvas?.getContext("2d");
        if (!canvas || !context) return;

        const image = new Image();
        image.onload = () => {
          canvas.width = image.width;
          canvas.height = image.height;
          context.drawImage(image, 0, 0, canvas.width, canvas.height);
          link.href = canvas.toDataURL(downloadFormat, 0.92);
          link.click();
        };
        image.src = frame.dataUrl;
        return;
      }
      link.click();
    },
    [downloadFormat]
  );

  return (
    <div className="min-h-screen bg-slate-50">
      <div className="mx-auto flex max-w-6xl flex-col gap-6 px-4 py-10 sm:px-6">
        <header className="flex flex-col gap-2 cursor-pointer" onClick={() => window.location.href = "https://6io.ru"}>
          <p className="text-sm font-semibold uppercase tracking-wide text-slate-500">
            Video Storyboard & Frame Extractor
          </p>
          <h1 className="text-3xl font-bold text-slate-900">
            Раскадровка видео прямо в браузере
          </h1>
          <p className="text-sm text-slate-600">
            Загрузите видео, укажите интервал и скачивайте понравившиеся кадры.
          </p>
        </header>

        <div className="grid gap-6 lg:grid-cols-[1.1fr_0.9fr]">
          <div className="flex flex-col gap-6">
            <VideoUploader
              onFileSelected={handleFileSelected}
              disabled={isLoading}
              helperText={helperText}
              error={error}
            />

            <section className="rounded-2xl border border-slate-200 bg-white p-6 shadow-sm">
              <div className="flex flex-col gap-3">
                <h2 className="text-lg font-semibold text-slate-900">Превью</h2>
                {!videoUrl && (
                  <p className="text-sm text-slate-500">
                    Загрузите видео, чтобы увидеть предпросмотр.
                  </p>
                )}
                {videoUrl && (
                  <video
                    ref={videoRef}
                    src={videoUrl}
                    controls
                    preload="metadata"
                    className="w-full rounded-lg border border-slate-200"
                    onLoadedMetadata={handleLoadedMetadata}
                  />
                )}
                {videoMeta && (
                  <div className="grid gap-2 text-xs text-slate-500 sm:grid-cols-3">
                    <div>Длительность: {formatTimestamp(videoMeta.duration)}</div>
                    <div>
                      Размер: {videoMeta.width}×{videoMeta.height}px
                    </div>
                    <div>
                      Кадров: {frames.length > 0 ? frames.length : "—"}
                    </div>
                  </div>
                )}
              </div>
            </section>
          </div>

          <div className="flex flex-col gap-6">
            <StoryboardControls
              stepSeconds={stepSeconds}
              onStepChange={setStepSeconds}
              onGenerate={generateStoryboard}
              disabled={!videoUrl}
              isLoading={isLoading}
            />

            <section className="rounded-2xl border border-slate-200 bg-white p-6 shadow-sm">
              <div className="flex flex-col gap-4">
                <h2 className="text-lg font-semibold text-slate-900">
                  Скачивание кадров
                </h2>
                <label className="text-sm font-medium text-slate-700">
                  <span className="mb-1 block">Формат файла</span>
                  <select
                    value={downloadFormat}
                    onChange={(event) =>
                      setDownloadFormat(event.target.value as "image/jpeg" | "image/png")
                    }
                    className="w-full rounded-lg border border-slate-200 bg-slate-50 px-3 py-2 text-sm text-slate-700"
                  >
                    <option value="image/jpeg">JPEG (быстрее, меньше вес)</option>
                    <option value="image/png">PNG (макс. качество)</option>
                  </select>
                </label>
                <p className="text-xs text-slate-500">
                  Выбранный формат применится к кнопке «Скачать» возле каждого кадра.
                </p>
              </div>
            </section>

            <section className="rounded-2xl border border-slate-200 bg-white p-6 shadow-sm">
              <div className="flex flex-col gap-3">
                <h2 className="text-lg font-semibold text-slate-900">Статус</h2>
                <p className="text-sm text-slate-600">
                  {isLoading
                    ? "Видео обрабатывается. Не закрывайте вкладку, пока идёт генерация кадров."
                    : "Готово к работе. Измените интервал или загрузите новое видео."}
                </p>
                {statusMessage && (
                  <p className="text-sm font-medium text-slate-900">
                    {statusMessage}
                  </p>
                )}
              </div>
            </section>
          </div>
        </div>

        <section className="flex flex-col gap-4">
          <div className="flex items-center justify-between">
            <h2 className="text-lg font-semibold text-slate-900">Раскадровка</h2>
            {frames.length > 0 && (
              <span className="text-xs text-slate-500">
                Всего кадров: {frames.length}
              </span>
            )}
          </div>
          <StoryboardGrid
            frames={frames}
            onDownload={handleDownload}
            formatTime={formatTimestamp}
          />
        </section>
      </div>
      {/* Скрытый canvas нужен для извлечения кадров. */}
      <canvas ref={canvasRef} className="hidden" />
    </div>
  );
};

export default App;
