/**
 * Video Storyboard & Frame Extractor
 * 
 * Главный компонент приложения для создания раскадровки видео.
 * 
 * Принцип работы:
 * 1. Пользователь загружает видеофайл (MP4/WebM)
 * 2. Видео загружается в <video> элемент
 * 3. При генерации раскадровки:
 *    - Создаётся <canvas> размером с видео
 *    - Для каждой временной метки устанавливается video.currentTime
 *    - После события 'seeked' кадр рисуется на canvas через drawImage
 *    - Canvas конвертируется в data URL через toDataURL
 * 4. Пользователь может скачать любой кадр в PNG или JPEG
 */

import { useState, useRef, useCallback } from 'react';
import type { AppState } from './types';
import {
  VideoUploader,
  VideoPlayer,
  FrameControls,
  FrameGrid,
  ProgressBar,
  ErrorMessage
} from './components';
import {
  validateVideoFile,
  getVideoInfo,
  extractFrames
} from './utils/videoUtils';

// Начальное состояние приложения
const initialState: AppState = {
  videoInfo: null,
  videoUrl: null,
  frames: [],
  frameInterval: 2,
  status: 'idle',
  progress: 0,
  error: null
};

function App() {
  // Состояние приложения
  const [state, setState] = useState<AppState>(initialState);
  
  // Ref на video элемент для извлечения кадров
  const videoRef = useRef<HTMLVideoElement>(null);

  /**
   * Обработка выбора файла
   */
  const handleFileSelect = useCallback(async (file: File) => {
    // Валидация файла
    const validationError = validateVideoFile(file);
    if (validationError) {
      setState(prev => ({
        ...prev,
        error: validationError,
        status: 'error'
      }));
      return;
    }

    // Освобождаем предыдущий URL
    if (state.videoUrl) {
      URL.revokeObjectURL(state.videoUrl);
    }

    // Устанавливаем статус загрузки
    setState(prev => ({
      ...prev,
      status: 'loading',
      error: null,
      frames: []
    }));

    try {
      // Создаём временный video элемент для получения метаданных
      const tempVideo = document.createElement('video');
      const videoInfo = await getVideoInfo(file, tempVideo);
      const videoUrl = tempVideo.src;

      setState(prev => ({
        ...prev,
        videoInfo,
        videoUrl,
        status: 'complete'
      }));
    } catch {
      setState(prev => ({
        ...prev,
        error: {
          code: 'PROCESSING_ERROR',
          message: 'Не удалось загрузить видео. Попробуйте другой файл.'
        },
        status: 'error'
      }));
    }
  }, [state.videoUrl]);

  /**
   * Изменение интервала между кадрами
   */
  const handleIntervalChange = useCallback((interval: number) => {
    setState(prev => ({ ...prev, frameInterval: interval }));
  }, []);

  /**
   * Генерация раскадровки
   */
  const handleExtract = useCallback(async () => {
    if (!videoRef.current || !state.videoInfo) return;

    setState(prev => ({
      ...prev,
      status: 'extracting',
      progress: 0,
      frames: []
    }));

    try {
      const frames = await extractFrames(
        videoRef.current,
        state.frameInterval,
        (progress) => {
          setState(prev => ({ ...prev, progress }));
        }
      );

      setState(prev => ({
        ...prev,
        frames,
        status: 'complete',
        progress: 100
      }));
    } catch {
      setState(prev => ({
        ...prev,
        error: {
          code: 'PROCESSING_ERROR',
          message: 'Ошибка при извлечении кадров. Попробуйте ещё раз.'
        },
        status: 'error'
      }));
    }
  }, [state.videoInfo, state.frameInterval]);

  /**
   * Сброс ошибки
   */
  const handleDismissError = useCallback(() => {
    setState(prev => ({
      ...prev,
      error: null,
      status: prev.videoInfo ? 'complete' : 'idle'
    }));
  }, []);

  /**
   * Сброс всего приложения
   */
  const handleReset = useCallback(() => {
    if (state.videoUrl) {
      URL.revokeObjectURL(state.videoUrl);
    }
    setState(initialState);
  }, [state.videoUrl]);

  return (
    <div className="min-h-screen bg-gray-50 dark:bg-gray-900">
      {/* Шапка */}
      <header className="bg-white dark:bg-gray-800 shadow-sm">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4">
          <div className="flex items-center justify-between">
            <div className="flex items-center gap-3 cursor-pointer" onClick={() => window.location.href = "https://6io.ru"}>
              <div className="w-10 h-10 bg-gradient-to-br from-blue-500 to-purple-600 rounded-lg flex items-center justify-center">
                <svg className="w-6 h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} 
                    d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z" />
                </svg>
              </div>
              <div>
                <h1 className="text-xl font-bold text-gray-900 dark:text-white">
                  Video Storyboard
                </h1>
                <p className="text-xs text-gray-500 dark:text-gray-400">
                  Frame Extractor
                </p>
              </div>
            </div>

            {/* Кнопка сброса */}
            {state.videoInfo && (
              <button
                onClick={handleReset}
                className="px-4 py-2 text-sm font-medium text-gray-600 dark:text-gray-300 
                  hover:text-gray-900 dark:hover:text-white hover:bg-gray-100 dark:hover:bg-gray-700 
                  rounded-lg transition-colors"
              >
                Загрузить другое видео
              </button>
            )}
          </div>
        </div>
      </header>

      {/* Основной контент */}
      <main className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        {/* Сообщение об ошибке */}
        {state.error && (
          <div className="mb-6">
            <ErrorMessage error={state.error} onDismiss={handleDismissError} />
          </div>
        )}

        {/* Загрузчик видео (показываем только если видео не загружено) */}
        {!state.videoInfo && (
          <div className="max-w-2xl mx-auto">
            <div className="text-center mb-8">
              <h2 className="text-2xl md:text-3xl font-bold text-gray-900 dark:text-white mb-3">
                Создайте раскадровку видео
              </h2>
              <p className="text-gray-600 dark:text-gray-400">
                Загрузите видео, чтобы извлечь кадры и создать визуальную раскадровку. 
                Вся обработка происходит в браузере — ваши файлы не загружаются на сервер.
              </p>
            </div>
            <VideoUploader 
              onFileSelect={handleFileSelect} 
              isDisabled={state.status === 'loading'}
            />
          </div>
        )}

        {/* Контент при загруженном видео */}
        {state.videoInfo && state.videoUrl && (
          <div className="space-y-6">
            {/* Видео и настройки */}
            <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
              {/* Видеоплеер */}
              <VideoPlayer 
                ref={videoRef}
                videoUrl={state.videoUrl} 
                videoInfo={state.videoInfo} 
              />

              {/* Панель настроек или прогресс */}
              {state.status === 'extracting' ? (
                <ProgressBar 
                  progress={state.progress} 
                  message="Извлечение кадров из видео..."
                />
              ) : (
                <FrameControls
                  frameInterval={state.frameInterval}
                  onIntervalChange={handleIntervalChange}
                  onExtract={handleExtract}
                  isExtracting={false}
                  videoDuration={state.videoInfo.duration}
                />
              )}
            </div>

            {/* Сетка кадров */}
            <FrameGrid frames={state.frames} />
          </div>
        )}

        {/* Индикатор загрузки видео */}
        {state.status === 'loading' && !state.videoInfo && (
          <div className="flex items-center justify-center py-12">
            <div className="text-center">
              <div className="w-12 h-12 border-4 border-blue-500 border-t-transparent rounded-full animate-spin mx-auto mb-4" />
              <p className="text-gray-600 dark:text-gray-400">Загрузка видео...</p>
            </div>
          </div>
        )}
      </main>

      {/* Футер */}
      <footer className="mt-auto py-6 text-center text-sm text-gray-500 dark:text-gray-400">
        <p>
          Все операции выполняются локально в вашем браузере. 
          Видео не загружается на внешние серверы.
        </p>
      </footer>
    </div>
  );
}

export default App;
