/**
 * Компонент карточки одного кадра
 */

import { useState } from 'react';
import type { ExtractedFrame, ImageFormat } from '../types';
import { formatTime, downloadFrame } from '../utils/videoUtils';

interface FrameCardProps {
  frame: ExtractedFrame;
  index: number;
}

export function FrameCard({ frame, index }: FrameCardProps) {
  const [isHovered, setIsHovered] = useState(false);
  const [showFormatMenu, setShowFormatMenu] = useState(false);

  const handleDownload = (format: ImageFormat) => {
    downloadFrame(frame, format);
    setShowFormatMenu(false);
  };

  return (
    <div 
      className="relative group bg-white dark:bg-gray-800 rounded-lg overflow-hidden shadow-md hover:shadow-xl transition-all duration-200"
      onMouseEnter={() => setIsHovered(true)}
      onMouseLeave={() => {
        setIsHovered(false);
        setShowFormatMenu(false);
      }}
    >
      {/* Изображение кадра */}
      <div className="aspect-video bg-gray-100 dark:bg-gray-900">
        <img 
          src={frame.dataUrl} 
          alt={`Кадр ${index + 1}`}
          className="w-full h-full object-cover"
          loading="lazy"
        />
      </div>

      {/* Оверлей с действиями */}
      <div className={`
        absolute inset-0 bg-black/50 flex items-center justify-center
        transition-opacity duration-200
        ${isHovered ? 'opacity-100' : 'opacity-0'}
      `}>
        <div className="relative">
          <button
            onClick={() => setShowFormatMenu(!showFormatMenu)}
            className="p-3 bg-white rounded-full shadow-lg hover:bg-gray-100 transition-colors"
            title="Скачать кадр"
          >
            <svg className="w-6 h-6 text-gray-800" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} 
                d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4" />
            </svg>
          </button>

          {/* Меню выбора формата */}
          {showFormatMenu && (
            <div className="absolute top-full mt-2 left-1/2 -translate-x-1/2 bg-white dark:bg-gray-800 rounded-lg shadow-xl overflow-hidden z-10">
              <button
                onClick={() => handleDownload('png')}
                className="block w-full px-4 py-2 text-sm text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 text-left"
              >
                PNG (лучшее качество)
              </button>
              <button
                onClick={() => handleDownload('jpeg')}
                className="block w-full px-4 py-2 text-sm text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 text-left"
              >
                JPEG (меньший размер)
              </button>
            </div>
          )}
        </div>
      </div>

      {/* Информация о кадре */}
      <div className="p-2 flex justify-between items-center">
        <span className="text-xs font-medium text-gray-500 dark:text-gray-400">
          #{index + 1}
        </span>
        <span className="text-xs text-gray-400 dark:text-gray-500 font-mono">
          {formatTime(frame.timestamp)}
        </span>
      </div>
    </div>
  );
}
