/**
 * Компонент панели настроек извлечения кадров
 */

interface FrameControlsProps {
  frameInterval: number;
  onIntervalChange: (interval: number) => void;
  onExtract: () => void;
  isExtracting: boolean;
  videoDuration: number;
}

export function FrameControls({ 
  frameInterval, 
  onIntervalChange, 
  onExtract, 
  isExtracting,
  videoDuration 
}: FrameControlsProps) {
  // Предустановленные интервалы
  const presets = [0.5, 1, 2, 5, 10];

  // Вычисляем количество кадров
  const estimatedFrames = Math.ceil(videoDuration / frameInterval);

  return (
    <div className="bg-white dark:bg-gray-800 rounded-xl p-4 md:p-6 shadow-md">
      <h3 className="text-lg font-semibold text-gray-800 dark:text-white mb-4">
        Настройки раскадровки
      </h3>

      {/* Выбор интервала */}
      <div className="mb-4">
        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          Интервал между кадрами
        </label>
        
        {/* Предустановленные значения */}
        <div className="flex flex-wrap gap-2 mb-3">
          {presets.map((preset) => (
            <button
              key={preset}
              onClick={() => onIntervalChange(preset)}
              disabled={isExtracting}
              className={`
                px-3 py-1.5 text-sm rounded-lg font-medium transition-colors
                ${frameInterval === preset
                  ? 'bg-blue-500 text-white'
                  : 'bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-600'
                }
                disabled:opacity-50 disabled:cursor-not-allowed
              `}
            >
              {preset} сек
            </button>
          ))}
        </div>

        {/* Слайдер для точной настройки */}
        <div className="flex items-center gap-4">
          <input
            type="range"
            min="0.5"
            max="30"
            step="0.5"
            value={frameInterval}
            onChange={(e) => onIntervalChange(parseFloat(e.target.value))}
            disabled={isExtracting}
            className="flex-1 h-2 bg-gray-200 dark:bg-gray-700 rounded-lg appearance-none cursor-pointer
              disabled:opacity-50 disabled:cursor-not-allowed
              [&::-webkit-slider-thumb]:w-4 [&::-webkit-slider-thumb]:h-4 
              [&::-webkit-slider-thumb]:bg-blue-500 [&::-webkit-slider-thumb]:rounded-full 
              [&::-webkit-slider-thumb]:appearance-none [&::-webkit-slider-thumb]:cursor-pointer"
          />
          <span className="text-sm font-medium text-gray-600 dark:text-gray-400 w-16 text-right">
            {frameInterval} сек
          </span>
        </div>
      </div>

      {/* Информация о количестве кадров */}
      <div className="bg-blue-50 dark:bg-blue-900/30 rounded-lg p-3 mb-4">
        <p className="text-sm text-blue-700 dark:text-blue-300">
          <span className="font-medium">Будет извлечено:</span> ~{estimatedFrames} кадр
          {estimatedFrames > 1 && estimatedFrames < 5 ? 'а' : estimatedFrames >= 5 ? 'ов' : ''}
        </p>
      </div>

      {/* Кнопка генерации */}
      <button
        onClick={onExtract}
        disabled={isExtracting}
        className={`
          w-full py-3 px-4 rounded-lg font-medium transition-all
          flex items-center justify-center gap-2
          ${isExtracting
            ? 'bg-gray-300 dark:bg-gray-600 text-gray-500 dark:text-gray-400 cursor-not-allowed'
            : 'bg-blue-500 hover:bg-blue-600 text-white shadow-md hover:shadow-lg'
          }
        `}
      >
        {isExtracting ? (
          <>
            <svg className="animate-spin h-5 w-5" viewBox="0 0 24 24">
              <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" fill="none" />
              <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z" />
            </svg>
            Извлечение...
          </>
        ) : (
          <>
            <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} 
                d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
            </svg>
            Сгенерировать раскадровку
          </>
        )}
      </button>
    </div>
  );
}
