/**
 * Компонент видеоплеера с превью
 * Отображает загруженное видео с базовыми контролами
 */

import { forwardRef } from 'react';
import type { VideoInfo } from '../types';
import { formatTime, formatFileSize } from '../utils/videoUtils';

interface VideoPlayerProps {
  videoUrl: string;
  videoInfo: VideoInfo;
  onTimeUpdate?: (currentTime: number) => void;
}

export const VideoPlayer = forwardRef<HTMLVideoElement, VideoPlayerProps>(
  function VideoPlayer({ videoUrl, videoInfo, onTimeUpdate }, ref) {
    return (
      <div className="bg-gray-900 rounded-xl overflow-hidden shadow-lg">
        {/* Видео плеер */}
        <div className="relative aspect-video bg-black">
          <video
            ref={ref}
            src={videoUrl}
            className="w-full h-full object-contain"
            controls
            onTimeUpdate={(e) => onTimeUpdate?.(e.currentTarget.currentTime)}
          />
        </div>

        {/* Информация о видео */}
        <div className="p-4 bg-gray-800">
          <h3 className="font-medium text-white truncate mb-2" title={videoInfo.name}>
            {videoInfo.name}
          </h3>
          
          <div className="flex flex-wrap gap-x-4 gap-y-1 text-sm text-gray-400">
            <span className="flex items-center gap-1">
              <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} 
                  d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
              </svg>
              {formatTime(videoInfo.duration)}
            </span>
            
            <span className="flex items-center gap-1">
              <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} 
                  d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
              </svg>
              {videoInfo.width}×{videoInfo.height}
            </span>
            
            <span className="flex items-center gap-1">
              <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} 
                  d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
              </svg>
              {formatFileSize(videoInfo.size)}
            </span>
          </div>
        </div>
      </div>
    );
  }
);
