/**
 * Компонент для загрузки видеофайла
 * Поддерживает drag & drop и выбор через диалог
 */

import { useRef, useState, useCallback } from 'react';

interface VideoUploaderProps {
  onFileSelect: (file: File) => void;
  isDisabled?: boolean;
}

export function VideoUploader({ onFileSelect, isDisabled = false }: VideoUploaderProps) {
  const [isDragging, setIsDragging] = useState(false);
  const inputRef = useRef<HTMLInputElement>(null);

  // Обработка drag событий
  const handleDragEnter = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    if (!isDisabled) setIsDragging(true);
  }, [isDisabled]);

  const handleDragLeave = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragging(false);
  }, []);

  const handleDragOver = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
  }, []);

  // Обработка drop события
  const handleDrop = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragging(false);

    if (isDisabled) return;

    const files = e.dataTransfer.files;
    if (files.length > 0) {
      onFileSelect(files[0]);
    }
  }, [isDisabled, onFileSelect]);

  // Обработка выбора файла через диалог
  const handleFileChange = useCallback((e: React.ChangeEvent<HTMLInputElement>) => {
    const files = e.target.files;
    if (files && files.length > 0) {
      onFileSelect(files[0]);
    }
    // Сбрасываем значение input для возможности повторной загрузки того же файла
    e.target.value = '';
  }, [onFileSelect]);

  // Открытие диалога выбора файла
  const handleClick = useCallback(() => {
    if (!isDisabled) {
      inputRef.current?.click();
    }
  }, [isDisabled]);

  return (
    <div
      className={`
        relative border-2 border-dashed rounded-xl p-8 md:p-12 
        transition-all duration-200 cursor-pointer
        ${isDragging 
          ? 'border-blue-500 bg-blue-50 dark:bg-blue-900/20' 
          : 'border-gray-300 dark:border-gray-600 hover:border-blue-400 dark:hover:border-blue-500'
        }
        ${isDisabled ? 'opacity-50 cursor-not-allowed' : ''}
      `}
      onDragEnter={handleDragEnter}
      onDragLeave={handleDragLeave}
      onDragOver={handleDragOver}
      onDrop={handleDrop}
      onClick={handleClick}
    >
      {/* Скрытый input для выбора файла */}
      <input
        ref={inputRef}
        type="file"
        accept="video/mp4,video/webm"
        onChange={handleFileChange}
        className="hidden"
        disabled={isDisabled}
      />

      <div className="flex flex-col items-center justify-center text-center">
        {/* Иконка */}
        <div className={`
          w-16 h-16 mb-4 rounded-full flex items-center justify-center
          ${isDragging 
            ? 'bg-blue-100 dark:bg-blue-800' 
            : 'bg-gray-100 dark:bg-gray-700'
          }
        `}>
          <svg 
            className={`w-8 h-8 ${isDragging ? 'text-blue-500' : 'text-gray-400'}`}
            fill="none" 
            stroke="currentColor" 
            viewBox="0 0 24 24"
          >
            <path 
              strokeLinecap="round" 
              strokeLinejoin="round" 
              strokeWidth={2} 
              d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12" 
            />
          </svg>
        </div>

        {/* Текст */}
        <h3 className="text-lg font-semibold text-gray-700 dark:text-gray-200 mb-2">
          {isDragging ? 'Отпустите файл' : 'Загрузите видео'}
        </h3>
        <p className="text-sm text-gray-500 dark:text-gray-400 mb-4">
          Перетащите файл сюда или нажмите для выбора
        </p>
        
        {/* Поддерживаемые форматы */}
        <div className="flex gap-2">
          <span className="px-3 py-1 text-xs font-medium bg-gray-100 dark:bg-gray-700 text-gray-600 dark:text-gray-300 rounded-full">
            MP4
          </span>
          <span className="px-3 py-1 text-xs font-medium bg-gray-100 dark:bg-gray-700 text-gray-600 dark:text-gray-300 rounded-full">
            WebM
          </span>
        </div>

        <p className="text-xs text-gray-400 dark:text-gray-500 mt-3">
          Максимальный размер: 500 MB
        </p>
      </div>
    </div>
  );
}
