/**
 * Утилиты для работы с видео и извлечения кадров
 */

import type { ExtractedFrame, VideoInfo, ImageFormat, AppError } from '../types';

/** Максимальный размер файла (500 MB) */
const MAX_FILE_SIZE = 500 * 1024 * 1024;

/** Поддерживаемые форматы видео */
const SUPPORTED_FORMATS = ['video/mp4', 'video/webm'];

/**
 * Валидация загруженного файла
 * @param file - Файл для проверки
 * @returns null если валидный, или объект ошибки
 */
export function validateVideoFile(file: File): AppError | null {
  // Проверка формата
  if (!SUPPORTED_FORMATS.includes(file.type)) {
    return {
      code: 'UNSUPPORTED_FORMAT',
      message: `Неподдерживаемый формат файла. Поддерживаются: MP4, WebM`
    };
  }

  // Проверка размера
  if (file.size > MAX_FILE_SIZE) {
    return {
      code: 'FILE_TOO_LARGE',
      message: `Файл слишком большой. Максимальный размер: ${formatFileSize(MAX_FILE_SIZE)}`
    };
  }

  return null;
}

/**
 * Получение информации о видео
 * @param file - Видеофайл
 * @param videoElement - HTML video элемент для загрузки
 * @returns Promise с информацией о видео
 */
export function getVideoInfo(file: File, videoElement: HTMLVideoElement): Promise<VideoInfo> {
  return new Promise((resolve, reject) => {
    // Создаём Object URL для видео
    const url = URL.createObjectURL(file);
    videoElement.src = url;

    // Обработчик загрузки метаданных
    const handleLoadedMetadata = () => {
      resolve({
        name: file.name,
        size: file.size,
        type: file.type,
        duration: videoElement.duration,
        width: videoElement.videoWidth,
        height: videoElement.videoHeight
      });
      cleanup();
    };

    // Обработчик ошибки
    const handleError = () => {
      URL.revokeObjectURL(url);
      reject(new Error('Не удалось загрузить видео'));
      cleanup();
    };

    const cleanup = () => {
      videoElement.removeEventListener('loadedmetadata', handleLoadedMetadata);
      videoElement.removeEventListener('error', handleError);
    };

    videoElement.addEventListener('loadedmetadata', handleLoadedMetadata);
    videoElement.addEventListener('error', handleError);

    // Начинаем загрузку
    videoElement.load();
  });
}

/**
 * Извлечение кадров из видео
 * 
 * Принцип работы:
 * 1. Устанавливаем video.currentTime на нужную временную метку
 * 2. Ждём события 'seeked' - видео перемотано на нужный кадр
 * 3. Рисуем текущий кадр на canvas с помощью drawImage
 * 4. Получаем data URL из canvas с помощью toDataURL
 * 
 * @param videoElement - HTML video элемент с загруженным видео
 * @param interval - Интервал между кадрами в секундах
 * @param onProgress - Callback для отслеживания прогресса
 * @returns Promise с массивом извлечённых кадров
 */
export async function extractFrames(
  videoElement: HTMLVideoElement,
  interval: number,
  onProgress?: (progress: number) => void
): Promise<ExtractedFrame[]> {
  const frames: ExtractedFrame[] = [];
  const duration = videoElement.duration;
  const totalFrames = Math.ceil(duration / interval);
  
  // Создаём canvas для захвата кадров
  // Canvas используется как буфер для рисования кадров видео
  const canvas = document.createElement('canvas');
  const ctx = canvas.getContext('2d');
  
  if (!ctx) {
    throw new Error('Не удалось создать canvas context');
  }

  // Устанавливаем размеры canvas равными размерам видео
  canvas.width = videoElement.videoWidth;
  canvas.height = videoElement.videoHeight;

  // Извлекаем кадры последовательно
  for (let i = 0; i < totalFrames; i++) {
    const timestamp = i * interval;
    
    // Не выходим за пределы видео
    if (timestamp > duration) break;

    // Перематываем видео на нужную временную метку и захватываем кадр
    const frame = await captureFrame(videoElement, canvas, ctx, timestamp);
    frames.push(frame);

    // Обновляем прогресс
    if (onProgress) {
      onProgress(Math.round(((i + 1) / totalFrames) * 100));
    }
  }

  return frames;
}

/**
 * Захват одного кадра из видео
 * 
 * @param video - Video элемент
 * @param canvas - Canvas для рисования
 * @param ctx - 2D контекст canvas
 * @param timestamp - Временная метка в секундах
 * @returns Promise с извлечённым кадром
 */
function captureFrame(
  video: HTMLVideoElement,
  canvas: HTMLCanvasElement,
  ctx: CanvasRenderingContext2D,
  timestamp: number
): Promise<ExtractedFrame> {
  return new Promise((resolve, reject) => {
    // Обработчик события seeked - видео перемотано на нужный кадр
    const handleSeeked = () => {
      try {
        // Очищаем canvas перед рисованием
        ctx.clearRect(0, 0, canvas.width, canvas.height);
        
        // Рисуем текущий кадр видео на canvas
        // drawImage копирует пиксели из видео в canvas
        ctx.drawImage(video, 0, 0, canvas.width, canvas.height);
        
        // Конвертируем canvas в data URL (base64 строка с изображением)
        // Используем PNG для лучшего качества
        const dataUrl = canvas.toDataURL('image/png');
        
        resolve({
          id: `frame-${timestamp.toFixed(2)}`,
          timestamp,
          dataUrl,
          width: canvas.width,
          height: canvas.height
        });
      } catch (error) {
        reject(error);
      }
      
      video.removeEventListener('seeked', handleSeeked);
    };

    const handleError = () => {
      reject(new Error(`Ошибка при захвате кадра на ${timestamp}s`));
      video.removeEventListener('seeked', handleSeeked);
      video.removeEventListener('error', handleError);
    };

    video.addEventListener('seeked', handleSeeked);
    video.addEventListener('error', handleError);
    
    // Устанавливаем временную метку - это вызовет перемотку видео
    video.currentTime = timestamp;
  });
}

/**
 * Скачивание кадра как изображения
 * 
 * @param frame - Извлечённый кадр
 * @param format - Формат изображения (png или jpeg)
 * @param quality - Качество для JPEG (0-1)
 */
export function downloadFrame(
  frame: ExtractedFrame, 
  format: ImageFormat = 'png',
  quality: number = 0.92
): void {
  // Создаём canvas для конвертации формата
  const canvas = document.createElement('canvas');
  const ctx = canvas.getContext('2d');
  
  if (!ctx) return;

  canvas.width = frame.width;
  canvas.height = frame.height;

  // Создаём изображение из data URL
  const img = new Image();
  img.onload = () => {
    ctx.drawImage(img, 0, 0);
    
    // Конвертируем в нужный формат
    const mimeType = format === 'jpeg' ? 'image/jpeg' : 'image/png';
    const dataUrl = canvas.toDataURL(mimeType, quality);
    
    // Создаём ссылку для скачивания
    const link = document.createElement('a');
    link.href = dataUrl;
    link.download = `frame_${formatTimestamp(frame.timestamp)}.${format}`;
    
    // Программно кликаем для скачивания
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };
  
  img.src = frame.dataUrl;
}

/**
 * Форматирование временной метки в строку MM:SS
 */
export function formatTimestamp(seconds: number): string {
  const mins = Math.floor(seconds / 60);
  const secs = Math.floor(seconds % 60);
  return `${mins.toString().padStart(2, '0')}_${secs.toString().padStart(2, '0')}`;
}

/**
 * Форматирование временной метки для отображения
 */
export function formatTime(seconds: number): string {
  const mins = Math.floor(seconds / 60);
  const secs = Math.floor(seconds % 60);
  return `${mins}:${secs.toString().padStart(2, '0')}`;
}

/**
 * Форматирование размера файла
 */
export function formatFileSize(bytes: number): string {
  if (bytes < 1024) return `${bytes} B`;
  if (bytes < 1024 * 1024) return `${(bytes / 1024).toFixed(1)} KB`;
  if (bytes < 1024 * 1024 * 1024) return `${(bytes / (1024 * 1024)).toFixed(1)} MB`;
  return `${(bytes / (1024 * 1024 * 1024)).toFixed(2)} GB`;
}
