/**
 * App.tsx — главный компонент приложения Video Storyboard & Frame Extractor.
 *
 * Управляет всем состоянием приложения:
 * - Загрузка видеофайла и чтение метаданных
 * - Извлечение кадров с заданным интервалом
 * - Отображение раскадровки
 *
 * Вся обработка видео происходит в браузере через <video> + <canvas>.
 */

import { useCallback, useState } from 'react';
import type { AppState, ExtractedFrame } from './types';
import { validateVideoFile, loadVideoInfo, extractFrames } from './utils';

import Header from './components/Header';
import DropZone from './components/DropZone';
import VideoPlayer from './components/VideoPlayer';
import ExtractionControls from './components/ExtractionControls';
import FrameGrid from './components/FrameGrid';
import ErrorMessage from './components/ErrorMessage';
import LoadingOverlay from './components/LoadingOverlay';

/** Начальное состояние приложения */
const initialState: AppState = {
  videoInfo: null,
  frames: [],
  status: 'idle',
  stepSeconds: 2,
  progress: 0,
  errorMessage: null,
};

function App() {
  const [state, setState] = useState<AppState>(initialState);

  /**
   * Обработка выбора файла пользователем.
   * 1. Валидирует файл (формат, размер)
   * 2. Создаёт Object URL для проигрывания
   * 3. Загружает метаданные (длительность, разрешение)
   */
  const handleFileSelect = useCallback(async (file: File) => {
    // Шаг 1: Валидация файла
    const error = validateVideoFile(file);
    if (error) {
      setState((prev) => ({ ...prev, errorMessage: error, status: 'error' }));
      return;
    }

    // Шаг 2: Переход в состояние загрузки
    setState((prev) => ({
      ...prev,
      status: 'loading',
      errorMessage: null,
      frames: [],
    }));

    try {
      // Шаг 3: Чтение метаданных видео
      const videoInfo = await loadVideoInfo(file);
      setState((prev) => ({
        ...prev,
        videoInfo,
        status: 'idle',
      }));
    } catch (err) {
      setState((prev) => ({
        ...prev,
        status: 'error',
        errorMessage: err instanceof Error ? err.message : 'Не удалось загрузить видео.',
      }));
    }
  }, []);

  /**
   * Запуск извлечения кадров из видео.
   * Использует скрытые <video> и <canvas> элементы для:
   * 1. Последовательного seek'а на нужные временные метки
   * 2. Рисования текущего кадра на canvas
   * 3. Экспорта canvas как Data URL (PNG)
   */
  const handleExtract = useCallback(async () => {
    if (!state.videoInfo) return;

    setState((prev) => ({
      ...prev,
      status: 'extracting',
      progress: 0,
      frames: [],
      errorMessage: null,
    }));

    try {
      const frames: ExtractedFrame[] = await extractFrames(
        state.videoInfo.objectUrl,
        state.videoInfo.duration,
        state.stepSeconds,
        (progress) => {
          setState((prev) => ({ ...prev, progress }));
        }
      );

      setState((prev) => ({
        ...prev,
        frames,
        status: 'done',
        progress: 100,
      }));
    } catch (err) {
      setState((prev) => ({
        ...prev,
        status: 'error',
        errorMessage: err instanceof Error ? err.message : 'Ошибка при извлечении кадров.',
      }));
    }
  }, [state.videoInfo, state.stepSeconds]);

  /** Смена шага извлечения кадров */
  const handleStepChange = useCallback((step: number) => {
    setState((prev) => ({ ...prev, stepSeconds: step }));
  }, []);

  /** Сброс — освобождение Object URL и возврат к начальному состоянию */
  const handleReset = useCallback(() => {
    if (state.videoInfo) {
      URL.revokeObjectURL(state.videoInfo.objectUrl);
    }
    setState(initialState);
  }, [state.videoInfo]);

  /** Закрытие сообщения об ошибке */
  const handleDismissError = useCallback(() => {
    setState((prev) => ({
      ...prev,
      errorMessage: null,
      status: prev.videoInfo ? 'idle' : 'idle',
    }));
  }, []);

  return (
    <div className="min-h-screen flex flex-col">
      <Header />

      <main className="flex-1 max-w-7xl w-full mx-auto px-4 sm:px-6 py-6 sm:py-8 space-y-6">
        {/* Сообщение об ошибке */}
        {state.errorMessage && (
          <ErrorMessage message={state.errorMessage} onDismiss={handleDismissError} />
        )}

        {/* Экран загрузки метаданных */}
        {state.status === 'loading' && <LoadingOverlay />}

        {/* Зона загрузки файла — показываем если нет видео */}
        {!state.videoInfo && state.status !== 'loading' && (
          <div className="space-y-4">
            <DropZone
              onFileSelect={handleFileSelect}
              disabled={false}
            />
            {/* Подсказка для пользователя */}
            <div className="text-center space-y-2">
              <p className="text-sm text-gray-500">
                Загрузите видео, чтобы начать создание раскадровки
              </p>
              <div className="flex flex-wrap justify-center gap-4 text-xs text-gray-600">
                <span className="flex items-center gap-1">
                  <span className="w-1.5 h-1.5 bg-violet-500 rounded-full" />
                  Работает в браузере
                </span>
                <span className="flex items-center gap-1">
                  <span className="w-1.5 h-1.5 bg-green-500 rounded-full" />
                  Файлы не загружаются на сервер
                </span>
                <span className="flex items-center gap-1">
                  <span className="w-1.5 h-1.5 bg-blue-500 rounded-full" />
                  Скачивайте кадры в PNG/JPEG
                </span>
              </div>
            </div>
          </div>
        )}

        {/* Видеоплеер и управление — показываем если видео загружено */}
        {state.videoInfo && (
          <>
            <VideoPlayer videoInfo={state.videoInfo} onReset={handleReset} />
            <ExtractionControls
              stepSeconds={state.stepSeconds}
              onStepChange={handleStepChange}
              onExtract={handleExtract}
              duration={state.videoInfo.duration}
              status={state.status}
              progress={state.progress}
            />
          </>
        )}

        {/* Сетка кадров */}
        {state.frames.length > 0 && (
          <FrameGrid frames={state.frames} stepSeconds={state.stepSeconds} />
        )}
      </main>

      {/* Футер */}
      <footer className="border-t border-gray-800/50 py-4 text-center text-xs text-gray-600">
        Video Storyboard & Frame Extractor · Все данные обрабатываются локально в вашем браузере
      </footer>
    </div>
  );
}

export default App;
