/**
 * Компонент DropZone — зона загрузки видеофайла.
 *
 * Поддерживает:
 * - Drag & Drop файлов
 * - Клик для выбора файла через диалог
 * - Валидация формата и размера
 */

import { useCallback, useRef, useState } from 'react';

interface DropZoneProps {
  /** Обработчик выбранного файла */
  onFileSelect: (file: File) => void;
  /** Приложение в процессе загрузки */
  disabled?: boolean;
}

const DropZone: React.FC<DropZoneProps> = ({ onFileSelect, disabled }) => {
  const [isDragging, setIsDragging] = useState(false);
  const inputRef = useRef<HTMLInputElement>(null);

  const handleDragOver = useCallback(
    (e: React.DragEvent) => {
      e.preventDefault();
      if (!disabled) setIsDragging(true);
    },
    [disabled]
  );

  const handleDragLeave = useCallback((e: React.DragEvent) => {
    e.preventDefault();
    setIsDragging(false);
  }, []);

  const handleDrop = useCallback(
    (e: React.DragEvent) => {
      e.preventDefault();
      setIsDragging(false);
      if (disabled) return;

      const file = e.dataTransfer.files[0];
      if (file) onFileSelect(file);
    },
    [disabled, onFileSelect]
  );

  const handleClick = useCallback(() => {
    if (!disabled) inputRef.current?.click();
  }, [disabled]);

  const handleChange = useCallback(
    (e: React.ChangeEvent<HTMLInputElement>) => {
      const file = e.target.files?.[0];
      if (file) onFileSelect(file);
      // Сбрасываем input, чтобы можно было загрузить тот же файл повторно
      e.target.value = '';
    },
    [onFileSelect]
  );

  return (
    <div
      onDragOver={handleDragOver}
      onDragLeave={handleDragLeave}
      onDrop={handleDrop}
      onClick={handleClick}
      className={`
        relative cursor-pointer rounded-2xl border-2 border-dashed
        transition-all duration-300 ease-out
        p-8 sm:p-12 text-center
        ${disabled ? 'opacity-50 cursor-not-allowed' : ''}
        ${
          isDragging
            ? 'border-violet-400 bg-violet-500/10 scale-[1.02]'
            : 'border-gray-700 bg-gray-900/50 hover:border-gray-500 hover:bg-gray-900/80'
        }
      `}
    >
      {/* Скрытый file input */}
      <input
        ref={inputRef}
        type="file"
        accept="video/mp4,video/webm"
        onChange={handleChange}
        className="hidden"
      />

      {/* Иконка загрузки */}
      <div className="mx-auto w-16 h-16 mb-4 rounded-full bg-gray-800 flex items-center justify-center">
        <svg
          className={`w-8 h-8 transition-colors ${isDragging ? 'text-violet-400' : 'text-gray-400'}`}
          fill="none"
          stroke="currentColor"
          viewBox="0 0 24 24"
        >
          <path
            strokeLinecap="round"
            strokeLinejoin="round"
            strokeWidth={1.5}
            d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12"
          />
        </svg>
      </div>

      <p className="text-base sm:text-lg font-medium text-gray-200 mb-1">
        {isDragging ? 'Отпустите файл для загрузки' : 'Перетащите видеофайл сюда'}
      </p>
      <p className="text-sm text-gray-500">
        или <span className="text-violet-400 underline underline-offset-2">выберите файл</span> на компьютере
      </p>
      <p className="text-xs text-gray-600 mt-3">
        Поддерживаемые форматы: MP4, WebM · Максимум 500 МБ
      </p>
    </div>
  );
};

export default DropZone;
