/**
 * Компонент ExtractionControls — панель управления извлечением кадров.
 *
 * Позволяет:
 * - Выбрать шаг извлечения (каждые N секунд)
 * - Запустить/перезапустить извлечение кадров
 * - Показывает ожидаемое количество кадров
 */

import type { ExtractionStatus } from '../types';
import { STEP_OPTIONS } from '../types';

interface ExtractionControlsProps {
  /** Текущий шаг в секундах */
  stepSeconds: number;
  /** Обработчик смены шага */
  onStepChange: (step: number) => void;
  /** Запуск извлечения */
  onExtract: () => void;
  /** Длительность видео в секундах */
  duration: number;
  /** Текущий статус */
  status: ExtractionStatus;
  /** Прогресс извлечения (0-100) */
  progress: number;
}

const ExtractionControls: React.FC<ExtractionControlsProps> = ({
  stepSeconds,
  onStepChange,
  onExtract,
  duration,
  status,
  progress,
}) => {
  // Ожидаемое количество кадров при текущих настройках
  const estimatedFrames = Math.ceil(duration / stepSeconds);
  const isProcessing = status === 'extracting';

  return (
    <div className="bg-gray-900/80 border border-gray-800 rounded-2xl p-4 sm:p-6 space-y-4">
      <h2 className="text-base font-semibold text-gray-200">Настройки раскадровки</h2>

      {/* Выбор шага */}
      <div className="space-y-2">
        <label className="block text-sm text-gray-400">
          Интервал между кадрами
        </label>
        <div className="flex flex-wrap gap-2">
          {STEP_OPTIONS.map((step) => (
            <button
              key={step}
              onClick={() => onStepChange(step)}
              disabled={isProcessing}
              className={`
                px-3 py-1.5 rounded-lg text-sm font-medium transition-all
                ${
                  stepSeconds === step
                    ? 'bg-violet-600 text-white shadow-lg shadow-violet-600/25'
                    : 'bg-gray-800 text-gray-300 hover:bg-gray-700 hover:text-white'
                }
                ${isProcessing ? 'opacity-50 cursor-not-allowed' : ''}
              `}
            >
              {step}с
            </button>
          ))}
        </div>
        <p className="text-xs text-gray-500">
          Будет извлечено ~{estimatedFrames} {getFrameWord(estimatedFrames)}
        </p>
      </div>

      {/* Кнопка запуска */}
      <button
        onClick={onExtract}
        disabled={isProcessing}
        className={`
          w-full py-3 rounded-xl font-semibold text-sm transition-all
          flex items-center justify-center gap-2
          ${
            isProcessing
              ? 'bg-gray-700 text-gray-400 cursor-not-allowed'
              : 'bg-gradient-to-r from-violet-600 to-indigo-600 text-white hover:from-violet-500 hover:to-indigo-500 shadow-lg shadow-violet-600/20 hover:shadow-violet-500/30 active:scale-[0.98]'
          }
        `}
      >
        {isProcessing ? (
          <>
            {/* Спиннер */}
            <svg className="animate-spin w-4 h-4" fill="none" viewBox="0 0 24 24">
              <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" />
              <path
                className="opacity-75"
                fill="currentColor"
                d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4z"
              />
            </svg>
            Извлечение… {progress}%
          </>
        ) : status === 'done' ? (
          <>
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
            </svg>
            Пересобрать раскадровку
          </>
        ) : (
          <>
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
            </svg>
            Создать раскадровку
          </>
        )}
      </button>

      {/* Прогресс-бар */}
      {isProcessing && (
        <div className="space-y-1">
          <div className="h-2 bg-gray-800 rounded-full overflow-hidden">
            <div
              className="h-full bg-gradient-to-r from-violet-500 to-indigo-500 rounded-full transition-all duration-300"
              style={{ width: `${progress}%` }}
            />
          </div>
          <p className="text-xs text-gray-500 text-center">
            Обработка видео… Не закрывайте страницу.
          </p>
        </div>
      )}
    </div>
  );
};

/**
 * Склонение слова «кадр» в зависимости от числа (русский язык)
 */
function getFrameWord(n: number): string {
  const abs = Math.abs(n) % 100;
  const lastDigit = abs % 10;
  if (abs > 10 && abs < 20) return 'кадров';
  if (lastDigit > 1 && lastDigit < 5) return 'кадра';
  if (lastDigit === 1) return 'кадр';
  return 'кадров';
}

export default ExtractionControls;
