/**
 * Компонент FrameCard — карточка одного извлечённого кадра.
 *
 * Отображает:
 * - Превью кадра
 * - Временную метку
 * - Кнопки скачивания (PNG / JPEG)
 */

import { useState } from 'react';
import type { ExtractedFrame, ImageFormat } from '../types';
import { downloadFrame, formatTimestamp } from '../utils';

interface FrameCardProps {
  frame: ExtractedFrame;
}

const FrameCard: React.FC<FrameCardProps> = ({ frame }) => {
  const [showActions, setShowActions] = useState(false);

  const handleDownload = (format: ImageFormat) => {
    downloadFrame(frame, format);
  };

  return (
    <div
      className="group relative bg-gray-900 rounded-xl overflow-hidden border border-gray-800 hover:border-gray-600 transition-all duration-200 hover:shadow-lg hover:shadow-black/30"
      onMouseEnter={() => setShowActions(true)}
      onMouseLeave={() => setShowActions(false)}
    >
      {/* Превью кадра */}
      <div className="relative aspect-video bg-black">
        <img
          src={frame.dataUrl}
          alt={`Кадр на ${formatTimestamp(frame.timestamp)}`}
          className="w-full h-full object-contain"
          loading="lazy"
        />

        {/* Оверлей с действиями — показывается при наведении (desktop) или всегда на mobile */}
        <div
          className={`
            absolute inset-0 bg-black/60 backdrop-blur-sm
            flex flex-col items-center justify-center gap-2
            transition-opacity duration-200
            ${showActions ? 'opacity-100' : 'opacity-0'}
            sm:opacity-0 sm:group-hover:opacity-100
          `}
        >
          <button
            onClick={() => handleDownload('png')}
            className="flex items-center gap-1.5 px-3 py-1.5 bg-white/10 hover:bg-white/20 rounded-lg text-sm text-white transition-colors"
          >
            <DownloadIcon />
            Скачать PNG
          </button>
          <button
            onClick={() => handleDownload('jpeg')}
            className="flex items-center gap-1.5 px-3 py-1.5 bg-white/10 hover:bg-white/20 rounded-lg text-sm text-white transition-colors"
          >
            <DownloadIcon />
            Скачать JPEG
          </button>
        </div>
      </div>

      {/* Информация о кадре */}
      <div className="px-3 py-2 flex items-center justify-between">
        <span className="text-xs font-mono text-gray-400">
          {formatTimestamp(frame.timestamp)}
        </span>
        <span className="text-xs text-gray-600">
          #{frame.index + 1}
        </span>
      </div>

      {/* Кнопки скачивания для мобильных (всегда видны) */}
      <div className="sm:hidden px-3 pb-3 flex gap-2">
        <button
          onClick={() => handleDownload('png')}
          className="flex-1 text-center text-xs py-1.5 bg-gray-800 hover:bg-gray-700 rounded-lg text-gray-300 transition-colors"
        >
          PNG
        </button>
        <button
          onClick={() => handleDownload('jpeg')}
          className="flex-1 text-center text-xs py-1.5 bg-gray-800 hover:bg-gray-700 rounded-lg text-gray-300 transition-colors"
        >
          JPEG
        </button>
      </div>
    </div>
  );
};

/** Иконка скачивания */
const DownloadIcon: React.FC = () => (
  <svg className="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
    <path
      strokeLinecap="round"
      strokeLinejoin="round"
      strokeWidth={2}
      d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-4l-4 4m0 0l-4-4m4 4V4"
    />
  </svg>
);

export default FrameCard;
