/**
 * Компонент FrameGrid — сетка всех извлечённых кадров.
 *
 * Отображает кадры в адаптивной сетке:
 * - 1 колонка на мобильных
 * - 2 колонки на планшетах
 * - 3-4 колонки на десктопе
 *
 * Также показывает общую статистику.
 */

import type { ExtractedFrame } from '../types';
import FrameCard from './FrameCard';

interface FrameGridProps {
  frames: ExtractedFrame[];
  stepSeconds: number;
}

const FrameGrid: React.FC<FrameGridProps> = ({ frames, stepSeconds }) => {
  if (frames.length === 0) return null;

  return (
    <div className="space-y-4">
      {/* Заголовок с информацией */}
      <div className="flex flex-wrap items-center justify-between gap-2 px-1">
        <h2 className="text-base font-semibold text-gray-200">
          Раскадровка
          <span className="ml-2 text-sm font-normal text-gray-500">
            {frames.length} {getFrameWord(frames.length)} · шаг {stepSeconds}с
          </span>
        </h2>
      </div>

      {/* Адаптивная сетка кадров */}
      <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-3 sm:gap-4">
        {frames.map((frame) => (
          <FrameCard key={frame.index} frame={frame} />
        ))}
      </div>
    </div>
  );
};

/** Склонение слова «кадр» */
function getFrameWord(n: number): string {
  const abs = Math.abs(n) % 100;
  const lastDigit = abs % 10;
  if (abs > 10 && abs < 20) return 'кадров';
  if (lastDigit > 1 && lastDigit < 5) return 'кадра';
  if (lastDigit === 1) return 'кадр';
  return 'кадров';
}

export default FrameGrid;
