/**
 * Компонент VideoPlayer — проигрыватель загруженного видео.
 *
 * Отображает:
 * - Нативный HTML5 <video> плеер
 * - Информацию о файле (имя, размер, разрешение, длительность)
 * - Кнопку для загрузки нового файла
 */

import type { VideoInfo } from '../types';
import { formatFileSize, formatTimestamp } from '../utils';

interface VideoPlayerProps {
  videoInfo: VideoInfo;
  /** Сброс видео и возврат к начальному экрану */
  onReset: () => void;
}

const VideoPlayer: React.FC<VideoPlayerProps> = ({ videoInfo, onReset }) => {
  return (
    <div className="space-y-4">
      {/* Плеер */}
      <div className="rounded-2xl overflow-hidden bg-black shadow-2xl shadow-black/50">
        {/**
         * Нативный <video> элемент с controls.
         * Используем objectUrl как src — это Blob URL из URL.createObjectURL().
         * preload="metadata" — загружаем только метаданные, не всё видео.
         */}
        <video
          src={videoInfo.objectUrl}
          controls
          preload="metadata"
          className="w-full max-h-[60vh] object-contain"
        />
      </div>

      {/* Информация о файле */}
      <div className="flex flex-wrap items-center justify-between gap-3 px-1">
        <div className="flex flex-wrap items-center gap-x-4 gap-y-1 text-sm text-gray-400">
          <span className="font-medium text-gray-200 truncate max-w-[200px] sm:max-w-[300px]" title={videoInfo.name}>
            {videoInfo.name}
          </span>
          <span>{formatFileSize(videoInfo.size)}</span>
          <span>{videoInfo.width}×{videoInfo.height}</span>
          <span>{formatTimestamp(videoInfo.duration)}</span>
        </div>

        <button
          onClick={onReset}
          className="text-sm text-gray-400 hover:text-white transition-colors flex items-center gap-1.5"
        >
          <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path
              strokeLinecap="round"
              strokeLinejoin="round"
              strokeWidth={2}
              d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15"
            />
          </svg>
          Загрузить другое видео
        </button>
      </div>
    </div>
  );
};

export default VideoPlayer;
