/**
 * Утилиты для работы с видео и canvas.
 *
 * Вся обработка происходит в браузере:
 * 1. Создаём скрытый элемент <video>
 * 2. Перематываем его на нужную временную метку
 * 3. Рисуем текущий кадр на <canvas>
 * 4. Экспортируем canvas как Data URL (PNG/JPEG)
 */

import type {
  ExtractedFrame,
  ImageFormat,
  VideoInfo,
} from './types';
import {
  SUPPORTED_VIDEO_TYPES,
  MAX_FILE_SIZE,
} from './types';

/**
 * Валидация файла перед загрузкой.
 * Проверяет тип и размер файла.
 */
export function validateVideoFile(file: File): string | null {
  if (!SUPPORTED_VIDEO_TYPES.includes(file.type as (typeof SUPPORTED_VIDEO_TYPES)[number])) {
    return `Неподдерживаемый формат файла: ${file.type || 'неизвестный'}. Поддерживаются только MP4 и WebM.`;
  }
  if (file.size > MAX_FILE_SIZE) {
    const sizeMB = Math.round(file.size / (1024 * 1024));
    return `Файл слишком большой (${sizeMB} МБ). Максимальный размер — ${MAX_FILE_SIZE / (1024 * 1024)} МБ.`;
  }
  return null;
}

/**
 * Загружает метаданные видеофайла (длительность, размеры).
 *
 * Создаёт временный элемент <video>, назначает ему Object URL
 * и ждёт события loadedmetadata для получения информации.
 */
export function loadVideoInfo(file: File): Promise<VideoInfo> {
  return new Promise((resolve, reject) => {
    const video = document.createElement('video');
    const objectUrl = URL.createObjectURL(file);

    video.preload = 'metadata';
    video.muted = true;
    video.src = objectUrl;

    video.onloadedmetadata = () => {
      // Для получения корректных размеров видео нужно дождаться
      // события loadeddata, когда первый кадр доступен
      video.onloadeddata = () => {
        resolve({
          name: file.name,
          size: file.size,
          type: file.type,
          duration: video.duration,
          width: video.videoWidth,
          height: video.videoHeight,
          objectUrl,
        });
      };
    };

    video.onerror = () => {
      URL.revokeObjectURL(objectUrl);
      reject(new Error('Не удалось загрузить видео. Возможно, файл повреждён.'));
    };
  });
}

/**
 * Извлекает один кадр из видео на заданной временной метке.
 *
 * Алгоритм:
 * 1. Устанавливаем video.currentTime на нужную метку
 * 2. Ждём событие seeked (видео перемоталось)
 * 3. Рисуем текущий кадр видео на canvas через drawImage()
 * 4. Конвертируем canvas в Data URL
 */
function extractSingleFrame(
  video: HTMLVideoElement,
  canvas: HTMLCanvasElement,
  ctx: CanvasRenderingContext2D,
  timestamp: number,
  index: number
): Promise<ExtractedFrame> {
  return new Promise((resolve, reject) => {
    // Устанавливаем позицию воспроизведения
    video.currentTime = timestamp;

    video.onseeked = () => {
      try {
        // Устанавливаем размеры canvas равными размерам видео
        canvas.width = video.videoWidth;
        canvas.height = video.videoHeight;

        // Рисуем текущий кадр видео на canvas
        // drawImage принимает элемент видео как источник изображения
        ctx.drawImage(video, 0, 0, canvas.width, canvas.height);

        // Экспортируем canvas как PNG Data URL
        const dataUrl = canvas.toDataURL('image/png');

        resolve({
          index,
          timestamp,
          dataUrl,
          width: canvas.width,
          height: canvas.height,
        });
      } catch (err) {
        reject(new Error(`Ошибка при извлечении кадра на ${timestamp}с: ${err}`));
      }
    };

    video.onerror = () => {
      reject(new Error(`Ошибка перемотки видео на ${timestamp}с`));
    };
  });
}

/**
 * Извлекает все кадры из видео с заданным шагом.
 *
 * Создаёт скрытые <video> и <canvas> элементы,
 * последовательно перематывает видео и снимает кадры.
 * Вызывает onProgress для отображения прогресса пользователю.
 */
export async function extractFrames(
  objectUrl: string,
  duration: number,
  stepSeconds: number,
  onProgress: (progress: number) => void
): Promise<ExtractedFrame[]> {
  const frames: ExtractedFrame[] = [];

  // Создаём скрытый video-элемент для извлечения кадров
  const video = document.createElement('video');
  video.src = objectUrl;
  video.muted = true;
  video.preload = 'auto';

  // Ждём полной загрузки метаданных видео
  await new Promise<void>((resolve, reject) => {
    video.onloadeddata = () => resolve();
    video.onerror = () => reject(new Error('Не удалось загрузить видео для обработки.'));
  });

  // Создаём offscreen canvas для рендеринга кадров
  const canvas = document.createElement('canvas');
  const ctx = canvas.getContext('2d');
  if (!ctx) {
    throw new Error('Браузер не поддерживает Canvas 2D контекст.');
  }

  // Вычисляем временные метки для извлечения кадров
  const timestamps: number[] = [];
  for (let t = 0; t < duration; t += stepSeconds) {
    timestamps.push(Math.min(t, duration));
  }

  // Последовательно извлекаем каждый кадр
  // (параллельное извлечение невозможно — один video элемент не может
  //  обрабатывать несколько seek-операций одновременно)
  for (let i = 0; i < timestamps.length; i++) {
    const frame = await extractSingleFrame(video, canvas, ctx, timestamps[i], i);
    frames.push(frame);

    // Обновляем прогресс (0–100%)
    const progress = Math.round(((i + 1) / timestamps.length) * 100);
    onProgress(progress);
  }

  return frames;
}

/**
 * Скачивает кадр как файл изображения.
 *
 * Создаёт временную ссылку <a> с атрибутом download,
 * программно "кликает" по ней для начала скачивания.
 */
export function downloadFrame(
  frame: ExtractedFrame,
  format: ImageFormat = 'png'
): void {
  // Если нужен JPEG, перерисовываем canvas с нужным форматом
  let dataUrl = frame.dataUrl;
  if (format === 'jpeg') {
    const canvas = document.createElement('canvas');
    canvas.width = frame.width;
    canvas.height = frame.height;
    const ctx = canvas.getContext('2d');
    if (ctx) {
      const img = new Image();
      img.src = frame.dataUrl;
      // Для синхронного рисования — dataUrl уже загружен в памяти
      ctx.drawImage(img, 0, 0);
      dataUrl = canvas.toDataURL('image/jpeg', 0.92);
    }
  }

  // Создаём временную ссылку для скачивания
  const link = document.createElement('a');
  link.href = dataUrl;
  const timestamp = frame.timestamp.toFixed(1).replace('.', '_');
  link.download = `frame_${timestamp}s.${format}`;
  document.body.appendChild(link);
  link.click();
  document.body.removeChild(link);
}

/**
 * Форматирует секунды в читаемый вид (MM:SS.s)
 */
export function formatTimestamp(seconds: number): string {
  const mins = Math.floor(seconds / 60);
  const secs = (seconds % 60).toFixed(1);
  return `${mins.toString().padStart(2, '0')}:${secs.padStart(4, '0')}`;
}

/**
 * Форматирует размер файла в человекочитаемый вид
 */
export function formatFileSize(bytes: number): string {
  if (bytes < 1024) return `${bytes} Б`;
  if (bytes < 1024 * 1024) return `${(bytes / 1024).toFixed(1)} КБ`;
  return `${(bytes / (1024 * 1024)).toFixed(1)} МБ`;
}
